<?php

namespace Modules\ExternalAiChat\App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Services\Integration\IntegrationService;
use App\Traits\Api\ApiResponseTrait;
use App\Utils\AppStatic;
use Illuminate\Http\Request;
use Modules\ExternalAiChat\App\Http\Requests\ChatBotRequest;
use Modules\ExternalAiChat\App\Models\ChatBot;
use Modules\ExternalAiChat\App\resources\ChatBotResource;
use Modules\ExternalAiChat\App\Services\ChatBotService;

class ChatBotController extends Controller
{
    
    use ApiResponseTrait;

    /**
     * Display a listing of the resource.
     */
    public function index(ChatBotService $chatBotService, Request $request)
    {
        $data["list"] = $chatBotService->all();

        if($request->route()->getPrefix() === 'api') {
            return response()->json($data);
        }

        if($request->ajax()){
            return view('externalaichat::chat-bots.list', $data)->render();
        }
        
        return view('externalaichat::chat-bots.index', $data)->render();
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ChatBotRequest $chatBotRequest, ChatBotService $chatBotService)
    {
        $data   = $chatBotRequest->getValidatedData();
        $result = $chatBotService->store($data);

        return $this->sendResponse(appStatic()::SUCCESS, 'Successfully added the chat expert.', $result);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ChatBotService $chatBotService, string $id)
    {
        $data = $chatBotService->getChatBotById($id, true);

        return $this->sendResponse(appStatic()::SUCCESS, '', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ChatBotService $chatBotService, ChatBotRequest $chatBotRequest, string $id)
    {
        try {
            $result = findById(new ChatBot(), $id);
            $data   = $chatBotRequest->getValidatedData($id);
            $result = $chatBotService->update($result, $data);

            return $this->sendResponse(appStatic()::SUCCESS, 'Successfully updated the information.', $result);

        } catch(\Exception $err) {
            return $this->sendResponse(appStatic()::VALIDATION_ERROR, 'There is an error during updating the information.', [], errorArray($err));
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    # chat page
    public function chat($uuid, ChatBotService $chatBotService)
    {
        $chatbot = $chatBotService->getChatBotByUUID($uuid);
        return view('externalaichat::chat-bots.show', compact('chatbot'));
    }

    # chatbot details
    public function chatbotDetails($uuid, ChatBotService $chatBotService)
    {
        return new ChatBotResource($chatBotService->getChatBotByUUID($uuid));
    }

    # chatbotThreads
    public function chatbotThreads($uuid, ChatBotService $chatBotService)
    {
        $chatbot             = $chatBotService->getChatBotByUUID($uuid);
        $chatbotThreads      = $chatBotService->getOrCreateChatBotThreads($chatbot);

        $firstThread         = $chatbotThreads->first();
        $firstThreadMessages = $firstThread->chats()->oldest()->get();
        return [
            'chatbotThreads'      => view('externalaichat::chat-bots.chat-thread-list', compact('chatbotThreads'))->render(),
            'chatbotThreadsCount' => count($chatbotThreads),
            'firstThreadMessages' => view('externalaichat::chat-bots.chat-thread-messages', compact('firstThreadMessages'))->render()
        ];
    }

    # chatbotThreadMessages
    public function chatbotThreadMessages($id, ChatBotService $chatBotService)
    {
        $firstThreadMessages = $chatBotService->getChatBotThreadMessages($id);
        return view('externalaichat::chat-bots.chat-thread-messages', compact('firstThreadMessages'))->render();
    }

    # new message
    public function chatbotThreadNewMessage(Request $request, ChatBotService $chatBotService, AppStatic $appStatic)
    {
        try {
            $chatThread        = $chatBotService->getChatBotThreadById($request->chat_bot_thread_id);
            $chatBotService->createChatbotThreadMessage($chatThread, null, $request->prompt);

            return $this->sendResponse(
                $appStatic::SUCCESS_WITH_DATA,
                localize("Chat thread conversation stored successfully."),
                ''
            );
        } catch (\Throwable $e) {

            return $this->sendResponse(
                $appStatic::VALIDATION_ERROR,
                $e->getMessage(),
                [],
                errorArray($e)
            );
        }
    }

    # chat with ai
    public function conversation(Request $request, IntegrationService $integrationService, AppStatic $appStatic, ChatBotService $chatBotService)
    {
        try {
            $chatThread        = $chatBotService->getChatBotThreadById($request->chat_bot_thread_id);
            // \Log::info('ChatBotController@conversation: chat_bot_thread_id=' . $request->chat_bot_thread_id . ', chat_bot_id=' . $chatThread->chat_bot_id);

            $request->merge([
                'stream'          => true,
                'chat_bot_id'     => $chatThread->chat_bot_id,
                'content_purpose' => $request->content_purpose ?? $appStatic::PURPOSE_CHAT,
            ]);

            setChatBot();
            setChatThreadId();
            setChatExpertId();

            $response   = [];
            $sessionLab = sessionLab();
            session()->forget($sessionLab::SESSION_OPEN_AI_ERROR);

            $response['title'] = $request->message;
            $engine            = $request["content_purpose"] == $appStatic::PURPOSE_CHAT ?  aiChatEngine() : $appStatic::ENGINE_OPEN_AI;

            // Regular Chats/ Streaming
            return $integrationService->contentGenerator($engine, $request);
        } catch (\Throwable $e) {

            wLog("Failed to store chat thread conversation", errorArray($e));

            return $this->streamErrorResponse($e->getMessage());
        }
    }

    # save response
    public function saveResponse(Request $request, ChatBotService $chatBotService, AppStatic $appStatic)
    {
        try {
            $chatThread        = $chatBotService->getChatBotThreadById($request->chat_bot_thread_id);
            $chatBotService->createChatbotThreadMessage($chatThread, $request->response, null); 
            return $this->sendResponse(
                $appStatic::SUCCESS_WITH_DATA,
                localize("Chat thread conversation stored successfully."),
                ''
            );
        } catch (\Throwable $e) {

            return $this->sendResponse(
                $appStatic::VALIDATION_ERROR,
                $e->getMessage(),
                [],
                errorArray($e)
            );
        }
    }
    
    # chatbotNewThread
    public function chatbotNewThread(Request $request, ChatBotService $chatBotService, AppStatic $appStatic)
    {  
        $chatbot = $chatBotService->getChatBotByUUID($request->uuid);

        $chatBotService->createChatbotThread($chatbot);
        
        $chatbotThreads      = $chatbot->threads()->get();
        return view('externalaichat::chat-bots.chat-thread-list', compact('chatbotThreads'))->render();
    }
}
